/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
*
* Copyright 2013 - 2020, nymea GmbH
* Contact: contact@nymea.io
*
* This file is part of maveod.
* This project including source code and documentation is protected by
* copyright law, and remains the property of nymea GmbH. All rights, including
* reproduction, publication, editing and translation, are reserved. The use of
* this project is subject to the terms of a license agreement to be concluded
* with nymea GmbH in accordance with the terms of use of nymea GmbH, available
* under https://nymea.io/license
*
* For any further details and any questions please contact us under
* contact@nymea.io or see our FAQ/Licensing Information on
* https://nymea.io/license/faq
*
* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

#include <QCoreApplication>
#include <QCommandLineParser>
#include <QCommandLineOption>

#include <unistd.h>

#include "maveod.h"
#include "application.h"
#include "loggingcategories.h"

static const char *const normal = "\033[0m";
static const char *const warning = "\033[33m";
static const char *const error = "\033[31m";

static void consoleLogHandler(QtMsgType type, const QMessageLogContext& context, const QString& message)
{
    switch (type) {
    case QtInfoMsg:
        fprintf(stdout, " I | %s: %s\n", context.category, message.toUtf8().data());
        break;
    case QtDebugMsg:
        fprintf(stdout, " I | %s: %s\n", context.category, message.toUtf8().data());
        break;
    case QtWarningMsg:
        fprintf(stdout, "%s W | %s: %s%s\n", warning, context.category, message.toUtf8().data(), normal);
        break;
    case QtCriticalMsg:
        fprintf(stdout, "%s C | %s: %s%s\n", error, context.category, message.toUtf8().data(), normal);
        break;
    case QtFatalMsg:
        fprintf(stdout, "%s F | %s: %s%s\n", error, context.category, message.toUtf8().data(), normal);
        break;
    }
    fflush(stdout);
}



int main(int argc, char *argv[])
{
    qInstallMessageHandler(consoleLogHandler);

    Application application(argc, argv);
    application.setApplicationName("maveod");
    application.setOrganizationName("nymea");
    application.setApplicationVersion(MAVEOD_VERSION_STRING);

    // Command line parser
    QCommandLineParser parser;
    parser.addHelpOption();
    parser.addVersionOption();
    parser.setApplicationDescription(QString("\nDaemon for the Maveo platform. This daemon allows to configure a wifi network with a bluetooth low energy connection.\n\nCopyright %1 2016 - 2020 nymea GmbH <contact@nymea.io>\nAll rights reserved.").arg(QChar(0xA9)));

    QCommandLineOption configurationOption(QStringList() << "c" << "configuration", QString("The filepath to the configuration file. Default is \"/etc/maveod/maveod.conf\"."), "CONFIG");
    configurationOption.setDefaultValue("/etc/maveod/maveod.conf");
    parser.addOption(configurationOption);
    parser.process(application);

    // Check user
    QString userName = qgetenv("USER");
    uint userId = getuid();
    if (userId != 0) {
        qCCritical(dcMaveod()) << "Access denied. Please start the daemon as root.\n";
        return EXIT_FAILURE;
    }

    qCDebug(dcMaveod()) << "======================================";
    qCDebug(dcMaveod()).noquote() << QString("Maveo daemon %1 started %2").arg(application.applicationVersion()).arg((userId != 0 ? "as user " + userName + "(" + QString::number(userId) + ")." : userName + "."));
    qCDebug(dcMaveod()) << "======================================";

    qCDebug(dcMaveod()) << "Start up procedure...";
    Maveod::instance()->setConfigurationFileName(parser.value(configurationOption));
    Maveod::instance()->run();

    return application.exec();
}
