/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
*
* Copyright 2013 - 2022, nymea GmbH
* Contact: contact@nymea.io
*
* This file is part of maveod.
* This project including source code and documentation is protected by
* copyright law, and remains the property of nymea GmbH. All rights, including
* reproduction, publication, editing and translation, are reserved. The use of
* this project is subject to the terms of a license agreement to be concluded
* with nymea GmbH in accordance with the terms of use of nymea GmbH, available
* under https://nymea.io/license
*
* For any further details and any questions please contact us under
* contact@nymea.io or see our FAQ/Licensing Information on
* https://nymea.io/license/faq
*
* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

#include "nymeadservice.h"
#include "loggingcategories.h"

#include <QTimer>

NymeadService::NymeadService(QObject *parent) :
    QObject(parent)
{
    // Check DBus connection
    if (!QDBusConnection::systemBus().isConnected()) {
        qCWarning(dcNymeaService()) << "System DBus not connected.";
        return;
    }

    // Get notification when nymead appears/disappears on DBus
    m_serviceWatcher = new QDBusServiceWatcher("io.guh.nymead", QDBusConnection::systemBus(), QDBusServiceWatcher::WatchForRegistration | QDBusServiceWatcher::WatchForUnregistration, this);
    connect(m_serviceWatcher, &QDBusServiceWatcher::serviceRegistered, this, &NymeadService::serviceRegistered);
    connect(m_serviceWatcher, &QDBusServiceWatcher::serviceUnregistered, this, &NymeadService::serviceUnregistered);

    initializeServices();
}

bool NymeadService::available() const
{
    return m_available;
}

void NymeadService::enableBluetooth(bool enable)
{
    if (!m_nymeadHardwareBluetoothInterface) {
        qCCritical(dcNymeaService()) << "Could not enable/disable bluetooth hardware resource. D-Bus interface not available.";
        return;
    }

    qCDebug(dcNymeaService()) << "Request nymea to" << (enable ? "enable" : "disable") << "bluetooth resources";

    QDBusMessage query = m_nymeadHardwareBluetoothInterface->call("EnableBluetooth", enable);
    if(query.type() != QDBusMessage::ReplyMessage) {
        qCWarning(dcNymeaService()) << "Could not enable/disable bluetooth on dbus:" << query.errorName() << query.errorMessage();
        return;
    }
}

void NymeadService::pushButtonPressed()
{
    if (!m_pushButtonAgent || !m_pushButtonAgent->initialized()) {
        qCCritical(dcNymeaService()) << "Could not press pushbutton. Pushbutton agent not available.";
        return;
    }

    qCDebug(dcNymeaService()) << "Pushbutton pressed. Send to nymead";
    m_pushButtonAgent->sendButtonPressed();
}

void NymeadService::setAvailable(bool available)
{
    if (m_available == available)
        return;

    if (available) {
        qCDebug(dcNymeaService())  << "Service is now available.";
    } else {
        qCWarning(dcNymeaService())  << "Service is not available any more.";
    }

    m_available = available;
    emit availableChanged(m_available);
}

bool NymeadService::init()
{
    qCDebug(dcNymeaService()) << "Start initializing nymead PushButton agent service";
    if (!m_pushButtonAgent)
        m_pushButtonAgent = new PushButtonAgent(this);

    if (!m_pushButtonAgent->initialized()) {
        if (!m_pushButtonAgent->init()) {
            qCWarning(dcNymeaService()) << "Could not initialize D-Bus PushButton agent.";
            m_pushButtonAgent->deleteLater();
            m_pushButtonAgent = nullptr;
            return false;
        }
    } else {
        qCDebug(dcNymeaService()) << "PushButton agent for nymead already initialized.";
    }

    if (!m_nymeadHardwareInterface) {
        qCDebug(dcNymeaService()) << "Creating nymead hardware manager service";
        m_nymeadHardwareInterface = new QDBusInterface("io.guh.nymead", "/io/guh/nymead/HardwareManager", "io.guh.nymead", QDBusConnection::systemBus(), this);
    }

    if (!m_nymeadHardwareInterface->isValid()) {
        qCWarning(dcNymeaService()) << "Invalid D-Bus HardwareManager interface.";
        m_nymeadHardwareInterface->deleteLater();
        m_nymeadHardwareInterface = nullptr;
        return false;
    }

    if (!m_nymeadHardwareBluetoothInterface) {
        qCDebug(dcNymeaService()) << "Creating nymead HardwareManager BluetoothLE service";
        m_nymeadHardwareBluetoothInterface = new QDBusInterface("io.guh.nymead", "/io/guh/nymead/HardwareManager/BluetoothLEManager", "io.guh.nymead", QDBusConnection::systemBus(), this);
    }

    if (!m_nymeadHardwareBluetoothInterface->isValid()) {
        qCWarning(dcNymeaService()) << "Invalid D-Bus HardwareManager BluetoothLE interface.";
        m_nymeadHardwareBluetoothInterface->deleteLater();
        m_nymeadHardwareBluetoothInterface = nullptr;
        return false;
    }

    return true;
}

void NymeadService::serviceRegistered(const QString &serviceName)
{
    qCDebug(dcNymeaService()) << "Service registered" << serviceName;
    initializeServices();
}

void NymeadService::serviceUnregistered(const QString &serviceName)
{
    qCDebug(dcNymeaService()) << "Service unregistered" << serviceName;

    if (m_pushButtonAgent) {
        m_pushButtonAgent->deleteLater();
        m_pushButtonAgent = nullptr;
    }

    if (m_nymeadHardwareInterface) {
        m_nymeadHardwareInterface->deleteLater();
        m_nymeadHardwareInterface = nullptr;
    }

    if (m_nymeadHardwareBluetoothInterface) {
        m_nymeadHardwareBluetoothInterface->deleteLater();
        m_nymeadHardwareBluetoothInterface = nullptr;
    }

    setAvailable(false);
    m_initRetry = 0;
}

void NymeadService::initializeServices()
{
    if (m_available)
        return;

    if (init()) {
        m_initRetry = 0;
        qCDebug(dcNymeaService()) << "Initialized nymea D-Bus services successfully";
        setAvailable(true);
    } else {
        setAvailable(false);
        m_initRetry++;

        if (m_initRetry > m_initRetryLimit) {
            qCWarning(dcNymeaService()) << "Failed to init nymea D-Bus services after" << m_initRetry << "attempts. Giving up. The service seems not to be available.";
        } else {
            qCWarning(dcNymeaService()) << "Failed to init nymea D-Bus services. Retry" << m_initRetry << "/" << m_initRetryLimit << "in 5 seconds.";
            QTimer::singleShot(5000, this, &NymeadService::initializeServices);
        }

    }
}
