/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
*
* Copyright 2013 - 2024, nymea GmbH
* Contact: contact@nymea.io
*
* This file is part of nymea.
* This project including source code and documentation is protected by
* copyright law, and remains the property of nymea GmbH. All rights, including
* reproduction, publication, editing and translation, are reserved. The use of
* this project is subject to the terms of a license agreement to be concluded
* with nymea GmbH in accordance with the terms of use of nymea GmbH, available
* under https://nymea.io/license
*
* GNU Lesser General Public License Usage
* Alternatively, this project may be redistributed and/or modified under the
* terms of the GNU Lesser General Public License as published by the Free
* Software Foundation; version 3. This project is distributed in the hope that
* it will be useful, but WITHOUT ANY WARRANTY; without even the implied
* warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
* Lesser General Public License for more details.
*
* You should have received a copy of the GNU Lesser General Public License
* along with this project. If not, see <https://www.gnu.org/licenses/>.
*
* For any further details and any questions please contact us under
* contact@nymea.io or see our FAQ/Licensing Information on
* https://nymea.io/license/faq
*
* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

#include "chargingsessionsjsonhandler.h"
#include "processreply.h"

#include <QLoggingCategory>
Q_DECLARE_LOGGING_CATEGORY(dcChargingSessions)

ChargingSessionsJsonHandler::ChargingSessionsJsonHandler(ChargingSessionsManager *sessionManager, QObject *parent)
    : JsonHandler{parent},
    m_sessionManager{sessionManager}
{
    qCDebug(dcChargingSessions()) << "Creating charging sessions JSON RPC handler";

    registerEnum<ChargingSessionsManager::ChargingSessionsError>();

    registerObject<ChargingSessionsConfiguration>();

    QVariantMap params, returns;
    QString description;

    params.clear(); returns.clear();
    description = "Generate a report of all charging sessions and send them to the configured email.";
    params.insert("o:carThingIds", QVariantList() << enumValueName(Uuid));
    returns.insert("chargingSessionsError", enumRef<ChargingSessionsManager::ChargingSessionsError>());
    registerMethod("SendReport", description, params, returns, Types::PermissionScopeNone);

    params.clear(); returns.clear();
    description = "Get configuration for charging sessions.";
    returns.insert("configuration", objectRef<ChargingSessionsConfiguration>());
    registerMethod("GetConfiguration", description, params, returns, Types::PermissionScopeNone);

    params.clear(); returns.clear();
    description = "Set configuration for charging sessions.";
    params.insert("configuration", objectRef<ChargingSessionsConfiguration>());
    returns.insert("chargingSessionsError", enumRef<ChargingSessionsManager::ChargingSessionsError>());
    registerMethod("SetConfiguration", description, params, returns, Types::PermissionScopeNone);

    // Notifications
    params.clear();
    description = "Emitted whenever the configuration changes.";
    params.insert("configuration", objectRef<ChargingSessionsConfiguration>());
    registerNotification("ConfigurationChanged", description, params);

    connect(sessionManager, &ChargingSessionsManager::configurationChanged, this ,[this](){
        QVariantMap params;
        params.insert("configuration", pack<ChargingSessionsConfiguration>(m_sessionManager->configuration()));
        emit ConfigurationChanged(params);
    });
}

QString ChargingSessionsJsonHandler::name() const
{
    return "ChargingSessions";
}

JsonReply *ChargingSessionsJsonHandler::SendReport(const QVariantMap &params)
{
    QList<ThingId> carThingIds;
    foreach (const QString &carThingIdString, params.value("carThingIds").toStringList()) {
        carThingIds << ThingId(carThingIdString);
    }

    ProcessReply *processReply = m_sessionManager->sendReport(carThingIds);
    if (processReply->isFinished()) {
        QVariantMap returns;
        returns.insert("chargingSessionsError", enumValueName<ChargingSessionsManager::ChargingSessionsError>(processReply->error()));
        return createReply(returns);
    }

    JsonReply *jsonReply = createAsyncReply("SendReport");
    connect(processReply, &ProcessReply::finished, this, [jsonReply, processReply](){
        QVariantMap returns;
        returns.insert("chargingSessionsError", enumValueName<ChargingSessionsManager::ChargingSessionsError>(processReply->error()));
        jsonReply->setData(returns);
        emit jsonReply->finished();
    });

    return jsonReply;
}

JsonReply *ChargingSessionsJsonHandler::GetConfiguration(const QVariantMap &params)
{
    Q_UNUSED(params)

    QVariantMap returns;
    returns.insert("configuration", pack<ChargingSessionsConfiguration>(m_sessionManager->configuration()));
    return createReply(returns);
}

JsonReply *ChargingSessionsJsonHandler::SetConfiguration(const QVariantMap &params)
{
    ChargingSessionsConfiguration configuration = unpack<ChargingSessionsConfiguration>(params.value("configuration"));
    ChargingSessionsManager::ChargingSessionsError error = m_sessionManager->setConfiguration(configuration);

    QVariantMap returns;
    returns.insert("chargingSessionsError", enumValueName<ChargingSessionsManager::ChargingSessionsError>(error));
    return createReply(returns);
}
