// SPDX-License-Identifier: GPL-3.0-or-later

/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
*
* Copyright (C) 2013 - 2024, nymea GmbH
* Copyright (C) 2024 - 2025, chargebyte austria GmbH
*
* This file is part of nymea-energy-plugin-chargingsessions.
*
* nymea-energy-plugin-chargingsessions is free software: you can redistribute it and/or modify
* it under the terms of the GNU General Public License as published by
* the Free Software Foundation, either version 3 of the License, or
* (at your option) any later version.
*
* nymea-energy-plugin-chargingsessions is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
* GNU General Public License for more details.
*
* You should have received a copy of the GNU General Public License
* along with nymea-energy-plugin-chargingsessions. If not, see <https://www.gnu.org/licenses/>.
*
* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

#ifndef CHARGINGSESSIONSDATABASE_H
#define CHARGINGSESSIONSDATABASE_H

#include <QUuid>
#include <QObject>
#include <QSqlDatabase>
#include <QSqlError>
#include <QSqlRecord>
#include <QFutureWatcher>

#include <integrations/thing.h>

#include "processreply.h"

#define DB_VERSION 1

class DatabaseJob;
class FetchDataReply;

class Session {

    friend class ChargingSessionsDatabase;

public:
    QString sessionId() const {
        return m_sessionId;
    }
    QString chargerName() const {
        return m_chargerName;
    }
    QString chargerSerialNumber() const {
        return m_chargerSerialNumber;
    }
    QString carName() const {
        return m_carName;
    }
    QDateTime startTimestamp() const {
        return m_startTimestamp;
    }
    QDateTime endTimestamp() const {
        return m_endTimestamp;
    }
    double sessionEnergy() const {
        return m_sessionEnergy;
    }
    double energyStart() const {
        return m_energyStart;
    }
    double energyEnd() const {
        return m_energyEnd;
    }

private:
    Session() = default;

    QString m_sessionId;
    QString m_chargerName;
    QString m_chargerSerialNumber;
    QString m_carName;
    QDateTime m_startTimestamp;
    QDateTime m_endTimestamp;
    double m_sessionEnergy;
    double m_energyStart;
    double m_energyEnd;

};

class ChargingSessionsDatabase: public QObject
{
    Q_OBJECT
public:

    ChargingSessionsDatabase(const QString &databaseName, QObject *parent = nullptr);
    ~ChargingSessionsDatabase();

    QString databaseName() const;

    // Log the current session start and returns the new session ID. Session id is negative on error.
    void logStartSession(const ThingId &evChargerId, const QString &evChargerName, const QString &serialNumber,
                         const ThingId &carId, const QString &carName, const QDateTime &startDateTime, double energyStart = 0);

    void logEndSession(int sessionId, const ThingId &carId, const QString &carName, const QDateTime &endDateTime, double energyEnd = 0);

    FetchDataReply *fetchCarSessions(const ThingId &carThingId = ThingId());
    FetchDataReply *fetchRow(uint sessionId);

    void updateSessionEnergy(uint sessionId, double sessionEnergy, const QDateTime &lastUpdate);
    void updateTotalEnergyConsumed(uint sessionId, double totalEnergyConsumed, const QDateTime &lastUpdate);

    bool wipeDatabase();

signals:
    void databaseSessionAdded(const ThingId &evChargerId, uint sessionId);
    void databaseSessionUpdated(uint sessionId);
    void databaseSessionFinished(uint sessionId);
    void writeCsvReportFinished();
    void jobsRunningChanged();

private slots:
    void enqueJob(DatabaseJob *job);
    void processQueue();
    void handleJobFinished();

private:
    QString m_databaseName;
    QString m_connectionName;
    uint m_currentDbVersion = 0;
    QSqlDatabase m_db;
    QList<DatabaseJob*> m_jobQueue;
    bool m_initialized;
    DatabaseJob *m_currentJob = nullptr;
    QFutureWatcher<DatabaseJob*> m_jobWatcher;
    uint m_lastSessionId = 0;

    Session parseSession(const QSqlRecord &result);

    bool initDatabase();
    bool createTable(const QString &tableName, const QString &schema);
};

class DatabaseJob: public QObject
{
    Q_OBJECT
public:
    DatabaseJob(const QSqlDatabase &db, const QString &queryString, const QVariantList &bindValues = QVariantList()):
        m_db(db),
        m_queryString(queryString),
        m_bindValues(bindValues)
    {
    }

    QString executedQuery() const { return m_executedQuery; }
    QSqlError error() const { return m_error; }
    QList<QSqlRecord> results() const { return m_results; }

signals:
    void finished();

private:
    QSqlDatabase m_db;
    QString m_queryString;
    QVariantList m_bindValues;
    qint64 m_startTimestamp;

    QString m_executedQuery;
    QSqlError m_error;
    QList<QSqlRecord> m_results;

    friend class ChargingSessionsDatabase;
};


class FetchDataReply : public ProcessReply
{
    Q_OBJECT
    friend class ChargingSessionsDatabase;

public:
    QList<Session> sessions() const {
        return m_sessions;
    };

private:
    explicit FetchDataReply(QObject *parent = nullptr) :
        ProcessReply{parent}
    {
    }
    QList<Session> m_sessions;
};

#endif // CHARGINGSESSIONSDATABASE_H
