#!/bin/bash

DATABASE=$(pwd)/charging-sessions.sqlite
ROWCOUNT=500
CARUUID=""

#------------------------------------------------------------------------------------------
function usage() {
    cat <<EOF

This tool allows to add charging sessions into a database for benchmark tests.

IMPORTANT: make sure there is at least one valid session in the database.

The last valid session will be picked up and copied "n" times into the database. 
The session ID will be picked up and continued, this allowes to use the tool multiple times and let the database grow.

If no car UUID has been passed, the last one will be picked up.

Usage: $(basename $0) [OPTIONS]

OPTIONS:
  -d, --database [FILE]                The resulting database file path. Default is "${DATABASE}".
  -n, --row-count [COUNT]              The count of rows you want to create in the database. Default is "${ROWCOUNT}"
  -u, --car-uuid [UUID]                The amount of different cars you want to create in the database for. Default is "${CARCOUNT}"
  -h, --help                           Show this message

EOF
}

#------------------------------------------------------------------------------------------

while [ "$1" != "" ]; do
    case $1 in
        -d | --database )
            DATABASE=$2
            shift;;
        -n | --row-count )
            ROWCOUNT=$2
            shift;;
        -u | --car-uuid )
            CARUUID=$2
            shift;;
        -h | --help )
            usage && exit 0;;
        * )
            usage && exit 1;;
    esac
    shift
done

which sqlite3 &>/dev/null

if [ $? -ne 0 ]; then
    echo "Missing dependency: please install the sqlite3 tool by running"
    echo ""
    echo "sudo apt-get install sqlite3"
    echo ""
    exit 1
fi

LAST_SESSIONID=$(sqlite3 ${DATABASE} "SELECT id FROM chargingSessions ORDER BY id DESC LIMIT 1;")
LAST_CHARGERUUID=$(sqlite3 ${DATABASE} "SELECT chargerUuid FROM chargingSessions WHERE endTimestamp NOT NULL ORDER BY id DESC LIMIT 1;")
LAST_CHARGERNAME=$(sqlite3 ${DATABASE} "SELECT chargerName FROM chargingSessions WHERE endTimestamp NOT NULL ORDER BY id DESC LIMIT 1;")
LAST_CHARGERSERIAL=$(sqlite3 ${DATABASE} "SELECT chargerSerialNumber FROM chargingSessions WHERE endTimestamp NOT NULL ORDER BY id DESC LIMIT 1;")
LAST_CARUUID=$(sqlite3 ${DATABASE} "SELECT carUuid FROM chargingSessions WHERE endTimestamp NOT NULL ORDER BY id DESC LIMIT 1;")
LAST_CARNAME=$(sqlite3 ${DATABASE} "SELECT carNAME FROM chargingSessions WHERE endTimestamp NOT NULL ORDER BY id DESC LIMIT 1;")
LAST_STARTTIMESTAMP=$(sqlite3 ${DATABASE} "SELECT startTimestamp FROM chargingSessions WHERE endTimestamp NOT NULL ORDER BY id DESC LIMIT 1;")
LAST_ENDTIMESTAMP=$(sqlite3 ${DATABASE} "SELECT endTimestamp FROM chargingSessions WHERE endTimestamp NOT NULL ORDER BY id DESC LIMIT 1;")
LAST_UPDATE=$(sqlite3 ${DATABASE} "SELECT lastUpdate FROM chargingSessions WHERE endTimestamp NOT NULL ORDER BY id DESC LIMIT 1;")
LAST_SESSIONENERGY=$(sqlite3 ${DATABASE} "SELECT sessionEnergy FROM chargingSessions WHERE endTimestamp NOT NULL ORDER BY id DESC LIMIT 1;")
LAST_STARTENERGY=$(sqlite3 ${DATABASE} "SELECT energyStart FROM chargingSessions WHERE endTimestamp NOT NULL ORDER BY id DESC LIMIT 1;")
LAST_ENDENERGY=$(sqlite3 ${DATABASE} "SELECT energyEnd FROM chargingSessions WHERE endTimestamp NOT NULL ORDER BY id DESC LIMIT 1;")

if [ ! -z ${CARUUID} ]; then
    echo "Using car UUID ${CARUUID} instead of the last session car UUID $LAST_CARUUID"
    LAST_CARUUID=${CARUUID}
fi

#sqlite3 ${DATABASE} "SELECT * FROM chargingSessions WHERE endTimestamp NOT NULL ORDER BY id DESC LIMIT 1;"

echo "- The highest session ID: $LAST_SESSIONID"
echo "- Last charger UUID: $LAST_CHARGERUUID"
echo "- Last charger name: $LAST_CHARGERNAME"
echo "- Last charger serial number: $LAST_CHARGERSERIAL"
echo "- Car UUID: $LAST_CARUUID"
echo "- Car name: $LAST_CARNAME"
echo "- Start: $LAST_STARTTIMESTAMP"
echo "- End: $LAST_ENDTIMESTAMP"
echo "- Updated: $LAST_ENDTIMESTAMP"
echo "- Session energy: $LAST_SESSIONENERGY"
echo "- Start energy: $LAST_STARTENERGY"
echo "- End energy; $LAST_ENDENERGY"
echo 



echo "Start adding ${ROWCOUNT} charging session from row ${STARTROW} into database in ${DATABASE} using car UUID ${CARUUID} cars..."
for ((i=1; i<${ROWCOUNT}; i++)); do
    SESSIONID=$((${LAST_SESSIONID} + ${i}))
    echo "Adding session:" ${SESSIONID}

    QUERY=$(cat << EOM
        INSERT OR REPLACE INTO chargingSessions (chargerUuid, chargerName, chargerSerialNumber, carUuid, carName, startTimestamp, endTimestamp, lastUpdate, sessionEnergy, energyStart, energyEnd) 
        VALUES (
            "$LAST_CHARGERUUID",
            "$LAST_CHARGERNAME", 
            "$LAST_CHARGERSERIAL", 
            "$LAST_CARUUID", 
            "$LAST_CARNAME", 
            $LAST_STARTTIMESTAMP,  
            $LAST_ENDTIMESTAMP, 
            $LAST_UPDATE,
            $LAST_SESSIONENERGY,
            $LAST_STARTENERGY, 
            $LAST_ENDENERGY 
        );
EOM
)
    echo $QUERY
    sqlite3 ${DATABASE} "${QUERY}"
done