// SPDX-License-Identifier: GPL-3.0-or-later

/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
*
* Copyright (C) 2013 - 2024, nymea GmbH
* Copyright (C) 2024 - 2025, chargebyte austria GmbH
*
* This file is part of nymea-energy-plugin-chargingsessions.
*
* nymea-energy-plugin-chargingsessions is free software: you can redistribute it and/or modify
* it under the terms of the GNU General Public License as published by
* the Free Software Foundation, either version 3 of the License, or
* (at your option) any later version.
*
* nymea-energy-plugin-chargingsessions is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
* GNU General Public License for more details.
*
* You should have received a copy of the GNU General Public License
* along with nymea-energy-plugin-chargingsessions. If not, see <https://www.gnu.org/licenses/>.
*
* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

#ifndef CHARGINGSESSIONSMANAGER_H
#define CHARGINGSESSIONSMANAGER_H

#include <QObject>

// from libnymea
#include <integrations/thingmanager.h>

// from libnymea-energy
#include <energymanager.h>

#include "mailclient.h"
#include "chargingsessionsconfiguration.h"

#include <QHash>
#include <QFuture>

class Session;
class ProcessReply;
class FetchDataReply;
class ChargingSessionsDatabase;
class EnergyManagerDbusClient;

class ChargingSessionsManager : public QObject
{
    Q_OBJECT
public:
    enum ChargingSessionsError {
        ChargingSessionsErrorNoError,
        ChargingSessionsErrorInvalidEmail,
        ChargingSessionsErrorUnknownCarThingId,
        ChargingSessionsErrorConfigurationIncomplete,
        ChargingSessionsErrorSendEmailFailed,
        ChargingSessionsErrorInternalError
    };
    Q_ENUM(ChargingSessionsError)

    explicit ChargingSessionsManager(EnergyManager *energyManager, ThingManager *thingManager, QObject *parent = nullptr);

    ProcessReply *sendReport(const QList<ThingId> carThingIds);

    ChargingSessionsConfiguration configuration() const;
    ChargingSessionsManager::ChargingSessionsError setConfiguration(const ChargingSessionsConfiguration &configuration);
    ChargingSessionsDatabase *database() const;

signals:
    void configurationChanged();

private slots:
    void onThingAdded(Thing *thing);
    void onThingRemoved(const ThingId &thingId);
    void onThingStateValueChanged(const StateTypeId &stateTypeId, const QVariant &value, const QVariant &minValue, const QVariant &maxValue, const QVariantList &possibleValues);

    void onEvChargerPluggedInChanged(Thing *evCharger, bool pluggedIn);
    void onEvChargerSessionEnergyChanged(Thing *evCharger, double sessionEnergy);
    void onEvChargerTotalEnergyConsumedChanged(Thing *evCharger, double totalEnergyConsumed);

private:
    EnergyManager *m_energyManager = nullptr;
    ThingManager *m_thingManager = nullptr;
    EnergyManagerDbusClient *m_energyManagerClient = nullptr;

    MailClient *m_mailClient = nullptr;
    ChargingSessionsDatabase *m_database = nullptr;
    ChargingSessionsConfiguration m_configuration;

    Things m_evChargers;

    QHash<uint, double> m_sessionStartEnergy;
    QHash<Thing *, uint> m_activeSessions;

    ThingId getAssociatedCarId(const ThingId &evChargerId);

    QStringList m_reportFiles;
    ProcessReply *m_sendReportReply = nullptr;
    QNetworkReply *m_networkReply = nullptr;;

    QList<FetchDataReply *> m_pendingFetchReplies;
    QList<QFutureWatcher<bool> *> m_pendingWriteJobs;

    QFutureWatcher<bool> *writeCsvFile(const QString &reportFileName, const QList<Session> &sessions);
    void onWriteCsvFileFinished(const QString &reportFileName, QFutureWatcher<bool> *watcher);

    bool verifyCharger(Thing *thing);

    void startMonitoringThingStates(Thing *thing);
    void stopMonitoringThingStates(Thing *thing);
};


#endif // CHARGINGSESSIONSMANAGER_H
