// SPDX-License-Identifier: GPL-3.0-or-later

/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
*
* Copyright (C) 2013 - 2024, nymea GmbH
* Copyright (C) 2024 - 2025, chargebyte austria GmbH
*
* This file is part of nymea-energy-plugin-chargingsessions.
*
* nymea-energy-plugin-chargingsessions is free software: you can redistribute it and/or modify
* it under the terms of the GNU General Public License as published by
* the Free Software Foundation, either version 3 of the License, or
* (at your option) any later version.
*
* nymea-energy-plugin-chargingsessions is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
* GNU General Public License for more details.
*
* You should have received a copy of the GNU General Public License
* along with nymea-energy-plugin-chargingsessions. If not, see <https://www.gnu.org/licenses/>.
*
* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

#include "chargingsessionsdbusinterface.h"

#include "../chargingsessionsdatabase.h"
#include "../chargingsessionsmanager.h"

#include <QDBusConnectionInterface>
#include <QDBusError>
#include <QDBusMessage>
#include <QEventLoop>
#include <QLoggingCategory>
#include <QSet>
#include <QSharedPointer>
#include <QEventLoop>
#include <QDateTime>

Q_DECLARE_LOGGING_CATEGORY(dcChargingSessions)

static const QString kDbusService = QStringLiteral("io.nymea.energy.chargingsessions");
static const QString kDbusPath = QStringLiteral("/io/nymea/energy/chargingsessions");
static const QString kDbusInterface = QStringLiteral("io.nymea.energy.chargingsessions");

ChargingSessionsDBusInterface::ChargingSessionsDBusInterface(ChargingSessionsManager *manager, QObject *parent) :
    QObject(parent),
    m_manager(manager),
    m_connection(QDBusConnection::systemBus())
{
    if (!m_connection.isConnected()) {
        qCWarning(dcChargingSessions()) << "Cannot register DBus interface. System bus is not connected.";
        return;
    }

    if (!m_connection.registerService(kDbusService)) {
        QDBusConnectionInterface *interface = m_connection.interface();
        if (!interface || !interface->isServiceRegistered(kDbusService)) {
            qCWarning(dcChargingSessions()) << "Failed to register DBus service" << kDbusService << m_connection.lastError().message();
            return;
        }
    }

    if (!m_connection.registerObject(kDbusPath, this, QDBusConnection::ExportAllSlots)) {
        qCWarning(dcChargingSessions()) << "Failed to register DBus object at" << kDbusPath << m_connection.lastError().message();
    } else {
        qCDebug(dcChargingSessions()) << "Charging sessions DBus interface registered at" << kDbusService << kDbusPath << kDbusInterface;
    }
}

QVariantList ChargingSessionsDBusInterface::GetSessions(const QStringList &carThingIds, qlonglong startTimestamp, qlonglong endTimestamp)
{
    const bool localLoop = (message().service() == m_connection.baseService());
    if (!localLoop) {
        setDelayedReply(true);
    }

    const QDBusMessage requestMessage = message();

    if (!m_connection.isConnected()) {
        qCWarning(dcChargingSessions()) << "Cannot handle GetSessions. System bus not connected.";
        if (!localLoop) {
            sendErrorReply(requestMessage, QStringLiteral("System bus not connected"));
        }
        return QVariantList();
    }

    if (!m_manager || !m_manager->database()) {
        qCWarning(dcChargingSessions()) << "Cannot handle GetSessions. Database is not available.";
        if (!localLoop) {
            sendErrorReply(requestMessage, QStringLiteral("Charging sessions database is not available"));
        }
        return QVariantList();
    }

    if (startTimestamp > 0 && endTimestamp > 0 && startTimestamp > endTimestamp) {
        qCWarning(dcChargingSessions()) << "Cannot handle GetSessions. Invalid timerange:" << startTimestamp << endTimestamp;
        if (!localLoop) {
            sendErrorReply(requestMessage, QStringLiteral("Invalid timerange"));
        }
        return QVariantList();
    }

    QList<ThingId> thingIds;
    QSet<QString> seenIds;
    foreach (const QString &idString, carThingIds) {
        if (idString.isEmpty()) {
            continue;
        }

        if (seenIds.contains(idString)) {
            continue;
        }

        seenIds.insert(idString);
        thingIds.append(ThingId(idString));
    }

    struct PendingState {
        int pendingReplies = 0;
        bool errorSent = false;
        QList<Session> sessions;
        QVariantList serialized;
        QEventLoop *loop = nullptr;
    };

    QSharedPointer<PendingState> state = QSharedPointer<PendingState>::create();

    auto finalize = [this, requestMessage, state, localLoop]() {
        if (state->pendingReplies != 0) {
            return;
        }

        if (!state->errorSent) {
            foreach (const Session &session, state->sessions) {
                state->serialized.append(serializeSession(session));
            }

            if (!localLoop) {
                sendSessionsReply(requestMessage, state->serialized);
            }
        }

        if (state->loop) {
            state->loop->quit();
        }
    };

    auto handleFinished = [this, requestMessage, state, localLoop, finalize](FetchDataReply *reply){
        if (state->errorSent) {
            state->pendingReplies--;
            finalize();
            return;
        }

        if (reply->error() != ChargingSessionsManager::ChargingSessionsErrorNoError) {
            qCWarning(dcChargingSessions()) << "Fetching sessions from database failed. Returning error to DBus caller.";
            state->errorSent = true;
            if (!localLoop) {
                sendErrorReply(requestMessage, QStringLiteral("Failed to fetch sessions from database"));
            }
            state->pendingReplies--;
            finalize();
            return;
        }

        state->sessions.append(reply->sessions());
        state->pendingReplies--;
        finalize();
    };

    const QDateTime startDateTime = (startTimestamp > 0) ? QDateTime::fromSecsSinceEpoch(startTimestamp) : QDateTime();
    const QDateTime endDateTime = (endTimestamp > 0) ? QDateTime::fromSecsSinceEpoch(endTimestamp) : QDateTime();

    if (thingIds.isEmpty()) {
        state->pendingReplies = 1;
        FetchDataReply *reply = m_manager->database()->fetchCarSessions(ThingId(), startDateTime, endDateTime);
        connect(reply, &FetchDataReply::finished, this, [handleFinished, reply](){
            handleFinished(reply);
        });
    }
    else {
        state->pendingReplies = thingIds.count();
        foreach (const ThingId &thingId, thingIds) {
            FetchDataReply *reply = m_manager->database()->fetchCarSessions(thingId, startDateTime, endDateTime);
            connect(reply, &FetchDataReply::finished, this, [handleFinished, reply](){
                handleFinished(reply);
            });
        }
    }

    if (localLoop) {
        QEventLoop loop;
        state->loop = &loop;
        loop.exec();
        return state->serialized;
    }

    return QVariantList();
}

void ChargingSessionsDBusInterface::sendErrorReply(const QDBusMessage &requestMessage, const QString &message)
{
    QDBusMessage reply = requestMessage.createErrorReply(QDBusError::Failed, message);
    m_connection.send(reply);
}

void ChargingSessionsDBusInterface::sendSessionsReply(const QDBusMessage &requestMessage, const QVariantList &sessions)
{
    QDBusMessage reply = requestMessage.createReply(QVariant::fromValue(sessions));
    m_connection.send(reply);
}

QVariantMap ChargingSessionsDBusInterface::serializeSession(const Session &session) const
{
    QVariantMap map;
    map.insert(QStringLiteral("sessionId"), session.sessionId());
    map.insert(QStringLiteral("chargerName"), session.chargerName());
    map.insert(QStringLiteral("chargerSerialNumber"), session.chargerSerialNumber());
    map.insert(QStringLiteral("carName"), session.carName());
    map.insert(QStringLiteral("startTimestamp"), session.startTimestamp().toSecsSinceEpoch());
    map.insert(QStringLiteral("endTimestamp"), session.endTimestamp().toSecsSinceEpoch());
    map.insert(QStringLiteral("sessionEnergy"), session.sessionEnergy());
    map.insert(QStringLiteral("energyStart"), session.energyStart());
    map.insert(QStringLiteral("energyEnd"), session.energyEnd());
    return map;
}
