/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
*
* Copyright 2013 - 2023, nymea GmbH
* Contact: contact@nymea.io
*
* This file is part of nymea.
* This project including source code and documentation is protected by
* copyright law, and remains the property of nymea GmbH. All rights, including
* reproduction, publication, editing and translation, are reserved. The use of
* this project is subject to the terms of a license agreement to be concluded
* with nymea GmbH in accordance with the terms of use of nymea GmbH, available
* under https://nymea.io/license
*
* GNU Lesser General Public License Usage
* Alternatively, this project may be redistributed and/or modified under the
* terms of the GNU Lesser General Public License as published by the Free
* Software Foundation; version 3. This project is distributed in the hope that
* it will be useful, but WITHOUT ANY WARRANTY; without even the implied
* warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
* Lesser General Public License for more details.
*
* You should have received a copy of the GNU Lesser General Public License
* along with this project. If not, see <https://www.gnu.org/licenses/>.
*
* For any further details and any questions please contact us under
* contact@nymea.io or see our FAQ/Licensing Information on
* https://nymea.io/license/faq
*
* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

#ifndef SPOTMARKETMANAGER_H
#define SPOTMARKETMANAGER_H

#include <QDebug>
#include <QTimer>
#include <QObject>
#include <QDateTime>
#include <QNetworkAccessManager>

#include "../types/chargingschedule.h"
#include "spotmarketdataprovider.h"

class SpotMarketManager : public QObject
{
    Q_OBJECT
public:
    explicit SpotMarketManager(QNetworkAccessManager *networkManager, QObject *parent = nullptr);

    bool enabled() const;
    void setEnabled(bool enabled);

    bool available() const;

    SpotMarketProviderInfos availableProviders() const;

    SpotMarketDataProvider *currentProvider() const;
    QUuid currentProviderId() const;

    bool changeProvider(const QUuid &providerId = QUuid());
    bool registerProvider(SpotMarketDataProvider *provider);

    // If no QDate given, it returns weigthed over all score entries.
    // If QDate is valid, returns the list of weighted score entries for the given day (00:00 - 24:00)
    // For now we see one day as window of optimization, maybe we will change the hour
    // when a day starts, but we ignore prices of the next day until we reached the next day.
    ScoreEntries weightedScoreEntries(const QDate &date = QDate()) const;
    static ScoreEntries weightScoreEntries(const ScoreEntries &scoreEntries);

    TimeFrames scheduleCharingTimeForToday(const QDateTime &currentDateTime, const uint minutes, uint minimumScheduleDuration = 1, bool currentFrameLocked = false);

    // Returns a list of schedules with a total of the given minutes as cheap as possible and as continuouse as possible.
    // The minimum schedule duration is in minutes and makes sure we don't enable / disable to often.
    // If the current frame is locked, the scheduler will make sure that the current fram will remain the current frame,
    // even if it would be better to stop an continue i.e. 3 min.
    // This feature should be used if we are already charing and prefere to keep charging...
    static TimeFrames scheduleChargingTime(const QDateTime &currentDateTime, const ScoreEntries weightedScores, const uint minutes, uint minimumScheduleDuration = 1, bool currentFrameLocked = false);

    // Returns a list of time frames fused together if possible
    static TimeFrames fuseTimeFrames(const TimeFrames &timeFrames);
    static TimeFrames fusePartialTimeFrames(const TimeFrames &timeFrames);


signals:
    void currentProviderChanged(SpotMarketDataProvider *provider);
    void availableChanged(bool available);
    void enabledChanged(bool enabled);
    void availableProvidersChanged();

    void scoreEntriesUpdated();

private slots:
    void onProviderScoreEntriesChanged(const ScoreEntries &scoreEntries);

private:
    QNetworkAccessManager *m_networkManager = nullptr;
    SpotMarketDataProvider *m_currentProvider = nullptr;

    SpotMarketProviderInfos m_availableProviderInfos;
    QHash<QUuid, SpotMarketDataProvider *> m_availableProviders;

    bool m_enabled = false;

    QHash<QDate, ScoreEntries> m_weightedScores;


};

#endif // SPOTMARKETMANAGER_H
