// SPDX-License-Identifier: GPL-3.0-or-later

/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
*
* Copyright (C) 2013 - 2024, nymea GmbH
* Copyright (C) 2024 - 2025, chargebyte austria GmbH
*
* This file is part of nymea-energy-plugin-nymea.
*
* nymea-energy-plugin-nymea.s free software: you can redistribute it and/or modify
* it under the terms of the GNU General Public License as published by
* the Free Software Foundation, either version 3 of the License, or
* (at your option) any later version.
*
* nymea-energy-plugin-nymea.s distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
* GNU General Public License for more details.
*
* You should have received a copy of the GNU General Public License
* along with nymea-energy-plugin-nymea. If not, see <https://www.gnu.org/licenses/>.
*
* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

#ifndef CHARGINGINFO_H
#define CHARGINGINFO_H

#include <integrations/thing.h>

#include <QMetaObject>
#include <QTime>
#include <QDebug>

class ChargingInfo
{
    Q_GADGET
    Q_PROPERTY(QUuid evChargerId READ evChargerId WRITE setEvChargerId)
    Q_PROPERTY(QUuid assignedCarId READ assignedCarId WRITE setAssignedCarId USER true)
    Q_PROPERTY(ChargingMode chargingMode READ chargingMode WRITE setChargingMode)
    Q_PROPERTY(QDateTime endDateTime READ endDateTime WRITE setEndDateTime USER true)
    Q_PROPERTY(QList<int> repeatDays READ repeatDays WRITE setRepeatDays USER true)
    Q_PROPERTY(uint targetPercentage READ targetPercentage WRITE setTargetPercentage USER true)
    Q_PROPERTY(ChargingState chargingState READ chargingState)
    Q_PROPERTY(bool spotMarketChargingEnabled READ spotMarketChargingEnabled WRITE setSpotMarketChargingEnabled USER true)
    Q_PROPERTY(uint dailySpotMarketPercentage READ dailySpotMarketPercentage WRITE setDailySpotMarketPercentage USER true)

public:
    enum ChargingMode {
        ChargingModeNormal = 0, // implicit default
        ChargingModeEco,
        ChargingModeEcoWithTargetTime
    };
    Q_ENUM(ChargingMode)

    enum ChargingState {
        ChargingStateIdle,
        ChargingStateSurplusCharging,
        ChargingStateSpotMarketCharging,
        ChargingStateTimeRequirement
    };
    Q_ENUM(ChargingState)

    ChargingInfo(const ThingId &evChargerId = ThingId());

    ThingId evChargerId() const;
    void setEvChargerId(const ThingId &evChargerId);

    ThingId assignedCarId() const;
    void setAssignedCarId(const ThingId &assignedCarId);

    ChargingMode chargingMode() const;
    void setChargingMode(ChargingMode chargingMode);

    QDateTime endDateTime() const;
    void setEndDateTime(const QDateTime &endDateTime);

    QList<int> repeatDays() const;
    void setRepeatDays(const QList<int> &repeatDays);

    uint targetPercentage() const;
    void setTargetPercentage(uint targetPercentage);

    QLocale locale() const;
    void setLocale(const QLocale &locale);

    ChargingState chargingState() const;
    void setChargingState(ChargingState chargingState);

    bool spotMarketChargingEnabled() const;
    void setSpotMarketChargingEnabled(bool spotMarketChargingEnabled);

    uint dailySpotMarketPercentage() const;
    void setDailySpotMarketPercentage(uint dailySpotMarketPercentage);

    bool operator==(const ChargingInfo &other) const;
    bool operator!=(const ChargingInfo &other) const;

    QDateTime nextEndTime(const QDateTime &currentDateTime) const;

private:
    ThingId m_evChargerId;
    ThingId m_assignedCarId;
    ChargingMode m_chargingMode = ChargingModeNormal;
    QDateTime m_endDateTime = QDateTime();
    QList<int> m_repeatDays;
    uint m_targetPercentage = 80;
    QLocale m_locale; // For push notifications regarding this evcharger/car. May be removed in the future and only sent out text ids which are to be translated in the app
    ChargingState m_chargingState = ChargingStateIdle;
    bool m_spotMarketChargingEnabled = false;
    uint m_dailySpotMarketPercentage = 0;

};

class ChargingInfos: public QList<ChargingInfo>
{
    Q_GADGET
    Q_PROPERTY(int count READ count)
public:
    ChargingInfos();
    ChargingInfos(const QList<ChargingInfo> &other);
    ChargingInfos(std::initializer_list<ChargingInfo> args):QList(args) {}
    Q_INVOKABLE QVariant get(int index) const;
    Q_INVOKABLE void put(const QVariant &variant);
};
Q_DECLARE_METATYPE(ChargingInfos)

QDebug operator<<(QDebug dbg, const ChargingInfo &chargingInfo);

#endif // CHARGINGINFO_H
