// SPDX-License-Identifier: GPL-3.0-or-later

/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
*
* Copyright (C) 2013 - 2024, nymea GmbH
* Copyright (C) 2024 - 2025, chargebyte austria GmbH
*
* This file is part of nymea-energy-plugin-nymea.
*
* nymea-energy-plugin-nymea.s free software: you can redistribute it and/or modify
* it under the terms of the GNU General Public License as published by
* the Free Software Foundation, either version 3 of the License, or
* (at your option) any later version.
*
* nymea-energy-plugin-nymea.s distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
* GNU General Public License for more details.
*
* You should have received a copy of the GNU General Public License
* along with nymea-energy-plugin-nymea. If not, see <https://www.gnu.org/licenses/>.
*
* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

#include "spotmarketdataprovider.h"
#include "../plugininfo.h"

#include <QFile>
#include <QSettings>
#include <nymeasettings.h>

SpotMarketDataProvider::SpotMarketDataProvider(QNetworkAccessManager *networkManager, QObject *parent)
    : QObject{parent},
      m_networkManager{networkManager}
{
    m_cacheFilePath = NymeaSettings::cachePath() + "/nymea-energy-spotmarket-data.cache";
}

SpotMarketProviderInfo SpotMarketDataProvider::info() const
{
    return m_info;
}

bool SpotMarketDataProvider::enabled() const
{
    return m_enabled;
}

bool SpotMarketDataProvider::available() const
{
    return m_available;
}

ScoreEntries SpotMarketDataProvider::scoreEntries() const
{
    return m_scoreEntries;
}

void SpotMarketDataProvider::cacheDataEntries(const ScoreEntries &scoreEntries)
{
    QSettings settings(m_cacheFilePath, QSettings::IniFormat);
    qCDebug(dcNymeaEnergy()) << this << "update cache" << m_cacheFilePath;
    settings.beginWriteArray(info().name(), scoreEntries.count());
    for (int i = 0; i < scoreEntries.count(); i++) {
        settings.setArrayIndex(i);
        settings.setValue("startDateTime", scoreEntries.at(i).startDateTime());
        settings.setValue("endDateTime", scoreEntries.at(i).endDateTime());
        settings.setValue("value", scoreEntries.at(i).value());
    }
    settings.endArray();
}

ScoreEntries SpotMarketDataProvider::loadCachedDataEntries()
{
    ScoreEntries entries;
    QSettings settings(NymeaSettings::cachePath() + "/nymea-energy-spotmarket-data.cache", QSettings::IniFormat);
    qCDebug(dcNymeaEnergy()) << this << "loading data from cache" << m_cacheFilePath;
    int size = settings.beginReadArray(info().name());
    for (int i = 0; i < size; i++) {
        settings.setArrayIndex(i);
        ScoreEntry entry;
        entry.setStartDateTime(settings.value("startDateTime").toDateTime());
        entry.setEndDateTime(settings.value("endDateTime").toDateTime());
        entry.setValue(settings.value("value").toDouble());
        entries.append(entry);
    }
    settings.endArray();
    return entries;
}

bool SpotMarketDataProvider::clearCache()
{
    qCDebug(dcNymeaEnergy()) << this << "clear cache" << m_cacheFilePath;
    return QFile::remove(m_cacheFilePath);
}

QDebug operator<<(QDebug debug, const SpotMarketProviderInfo &providerInfo)
{
    QDebugStateSaver saver(debug);
    debug.nospace() << "SpotMarktProviderInfo(" << providerInfo.name() << ", " << providerInfo.country() << ", " << providerInfo.website() << ", id: " << providerInfo.providerId().toString() << ")";
    return debug;
}

QDebug operator<<(QDebug debug, SpotMarketDataProvider *provider)
{
    QDebugStateSaver saver(debug);
    debug.nospace() << "SpotMarktDataProvider(" << provider->info().name() << ", " << provider->info().country() << ")";
    return debug;
}
