/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
*
* Copyright 2013 - 2025, nymea GmbH
* Contact: contact@nymea.io
*
* This file is part of nymea.
* This project including source code and documentation is protected by
* copyright law, and remains the property of nymea GmbH. All rights, including
* reproduction, publication, editing and translation, are reserved. The use of
* this project is subject to the terms of a license agreement to be concluded
* with nymea GmbH in accordance with the terms of use of nymea GmbH, available
* under https://nymea.io/license
*
* For any further details and any questions please contact us under
* contact@nymea.io or see our FAQ/Licensing Information on
* https://nymea.io/license/faq
*
* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

#include "streamunlimitedrequest.h"

#include <network/networkaccessmanager.h>

#include <QNetworkReply>
#include <QJsonParseError>
#include <QJsonDocument>
#include <QUrlQuery>
#include <QLoggingCategory>

Q_DECLARE_LOGGING_CATEGORY(dcStreamUnlimited)

StreamUnlimitedGetRequest::StreamUnlimitedGetRequest(NetworkAccessManager *nam, const QHostAddress &hostAddress, int port, const QString &path, const QStringList &roles, QObject *parent):
    QObject(parent)
{
    QUrl url;
    url.setScheme("http");
    url.setHost(hostAddress.toString());
    url.setPort(port);
    url.setPath("/api/getData");

    QUrlQuery query;
    query.addQueryItem("path", path);
    query.addQueryItem("roles", roles.join(','));
    url.setQuery(query);

    QNetworkRequest request(url);

//    qCDebug(dcStreamUnlimited())<< "Get data request:" << url.toString();

    QNetworkReply *reply = nam->get(request);
    connect(reply, &QNetworkReply::finished, reply, &QNetworkReply::deleteLater);
    connect(reply, &QNetworkReply::finished, this, [=](){
        deleteLater();
        if (reply->error() != QNetworkReply::NoError) {
            qCWarning(dcStreamUnlimited()) << "Request to" << hostAddress.toString() << "failed:" << reply->errorString();
            emit error(reply->error());
            return;
        }

        QByteArray data = reply->readAll();
        QJsonParseError parseError;
        QJsonDocument jsonDoc = QJsonDocument::fromJson(data, &parseError);
        if (parseError.error != QJsonParseError::NoError) {
            qCWarning(dcStreamUnlimited()) << "Json parse error in reply from" << hostAddress.toString() << ":" << parseError.errorString();
            emit error(QNetworkReply::UnknownContentError);
            return;
        }

//        qCDebug(dcStreamUnlimited()) << "GetData reply:" << qUtf8Printable(jsonDoc.toJson());

        QVariantList resultList = jsonDoc.toVariant().toList();

        if (resultList.count() != roles.count()) {
            qCWarning(dcStreamUnlimited()) << "Unexpected result length!";
            emit error(QNetworkReply::UnknownContentError);
            return;
        }

        QVariantMap resultMap;
        for (int i = 0; i < roles.length(); i++) {
            resultMap.insert(roles.at(i), resultList.at(i));
        }
        emit finished(resultMap);
    });
}

StreamUnlimitedSetRequest::StreamUnlimitedSetRequest(NetworkAccessManager *nam, const QHostAddress &hostAddress, int port, const QString &path, const QString &role, const QVariant &value, QObject *parent, QNetworkAccessManager::Operation operation):
    QObject(parent)
{
    QUrl url;
    url.setScheme("http");
    url.setHost(hostAddress.toString());
    url.setPort(port);
    url.setPath("/api/setData");

    QNetworkReply *reply = nullptr;

    if (operation == QNetworkAccessManager::GetOperation) {
        QUrlQuery query;
        query.addQueryItem("path", path);
        query.addQueryItem("role", role);
#if QT_VERSION >= QT_VERSION_CHECK(6, 0, 0)
        if (value.typeId() == QMetaType::QString) {
#else
        if (value.type() == QVariant::String) {
#endif
            query.addQueryItem("value", value.toString());
        } else {
            query.addQueryItem("value", QJsonDocument::fromVariant(value).toJson(QJsonDocument::Compact));
        }
        url.setQuery(query);

        QNetworkRequest request(url);

        qCDebug(dcStreamUnlimited())<< "Set data request:" << url.toString();

        reply = nam->get(request);
    } else if (operation == QNetworkAccessManager::PostOperation) {

        QVariantMap payloadMap;
        payloadMap.insert("path", path);
        payloadMap.insert("role", role);
        payloadMap.insert("value", value);
        QByteArray payload = QJsonDocument::fromVariant(payloadMap).toJson(QJsonDocument::Compact);

        QNetworkRequest request(url);
        request.setHeader(QNetworkRequest::ContentTypeHeader, "application/json");

        qCDebug(dcStreamUnlimited()) << "Set data request:" << url.toString() << payload;

        reply = nam->post(request, payload);
    } else {
        qCWarning(dcStreamUnlimited()) << "Operation" << operation << "not supported for setData call";
        staticMetaObject.invokeMethod(this, "error", Qt::QueuedConnection, Q_ARG(QNetworkReply::NetworkError, QNetworkReply::OperationNotImplementedError));
        return;
    }

    connect(reply, &QNetworkReply::finished, reply, &QNetworkReply::deleteLater);
    connect(reply, &QNetworkReply::finished, this, [=](){
        deleteLater();
        if (reply->error() != QNetworkReply::NoError) {
            qCWarning(dcStreamUnlimited()) << "Request to" << hostAddress.toString() << "failed:" << reply->errorString() << reply->readAll();
            emit error(reply->error());
            return;
        }

        QByteArray data = reply->readAll();
        emit finished(data);
    });
}

StreamUnlimitedBrowseRequest::StreamUnlimitedBrowseRequest(NetworkAccessManager *nam, const QHostAddress &hostAddress, int port, const QString &path, const QStringList &roles, QObject *parent):
    QObject(parent)
{
    fetchBatch(nam, hostAddress, port, path, roles, 0, 29);
}

void StreamUnlimitedBrowseRequest::fetchBatch(NetworkAccessManager *nam, const QHostAddress &hostAddress, int port, const QString &path, const QStringList &roles, int from, int to)
{
    QUrl url;
    url.setScheme("http");
    url.setHost(hostAddress.toString());
    url.setPort(port);
    url.setPath("/api/getRows");

    QUrlQuery query;
    query.addQueryItem("path", path);
    query.addQueryItem("roles", roles.join(','));
    query.addQueryItem("from", QString::number(from));
    query.addQueryItem("to", QString::number(to));
    url.setQuery(query);

    QNetworkRequest request(url);

    qCDebug(dcStreamUnlimited()) << "Browse request:" << url.toString();

    QNetworkReply *reply = nam->get(request);
    connect(reply, &QNetworkReply::finished, reply, &QNetworkReply::deleteLater);
    connect(reply, &QNetworkReply::finished, this, [=](){
        if (reply->error() != QNetworkReply::NoError) {
            qCWarning(dcStreamUnlimited()) << "Request to" << hostAddress.toString() << "failed:" << reply->errorString();
            emit error(reply->error());
            deleteLater();
            return;
        }

        QByteArray data = reply->readAll();
        QJsonParseError parseError;
        QJsonDocument jsonDoc = QJsonDocument::fromJson(data, &parseError);
        if (parseError.error != QJsonParseError::NoError) {
            qCWarning(dcStreamUnlimited()) << "Json parse error in reply from" << hostAddress.toString() << ":" << parseError.errorString();
            emit error(QNetworkReply::UnknownContentError);
            deleteLater();
            return;
        }

        QVariantMap response = jsonDoc.toVariant().toMap();

        m_rowCache.append(response.value("rows").toList());
        qCDebug(dcStreamUnlimited()) <<  "Browse data row count" << m_rowCache.count();

        // If there's more data, loop again...
        if (response.value("rows").toList().count() == 30) {
            fetchBatch(nam, hostAddress, port, path, roles, m_rowCache.count(), m_rowCache.count() + 29);
            return;
        }

        // else, we're done. Update the result will all the rows
        response["rows"] = m_rowCache;

        emit finished(response);
        deleteLater();
    });
}

