// SPDX-License-Identifier: GPL-3.0-or-later

/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
*
* Copyright (C) 2013 - 2024, nymea GmbH
* Copyright (C) 2024 - 2025, chargebyte austria GmbH
*
* This file is part of nymea-update-plugin-mender.
*
* nymea-update-plugin-mender is free software: you can redistribute it and/or modify
* it under the terms of the GNU General Public License as published by
* the Free Software Foundation, either version 3 of the License, or
* (at your option) any later version.
*
* nymea-update-plugin-mender is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
* GNU General Public License for more details.
*
* You should have received a copy of the GNU General Public License
* along with nymea-update-plugin-mender. If not, see <https://www.gnu.org/licenses/>.
*
* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

#ifndef UPDATECONTROLLERMENDER_H
#define UPDATECONTROLLERMENDER_H

#include <QDir>
#include <QTimer>
#include <QObject>
#include <QProcess>
#include <QFileInfo>

#include "dbus/dbusagent.h"
#include "platform/platformupdatecontroller.h"

class UpdateControllerMender: public PlatformUpdateController
{
    Q_OBJECT
    Q_PLUGIN_METADATA(IID "io.nymea.PlatformUpdateController")
    Q_INTERFACES(PlatformUpdateController)

public:
    enum State {
        StateUnknown,
        StateIdle,
        StateSync,
        StateDownload,
        StateUpdateAvailable, // Custom state
        StateArtifactInstall,
        StateArtifactReboot,
        StateArtifactCommit,
        StateArtifactRollback,
        StateArtifactRollbackReboot,
        StateArtifactFailure
    };
    Q_ENUM(State)

    explicit UpdateControllerMender(QObject *parent = nullptr);

    bool updateManagementAvailable() const override;
    bool checkForUpdates() override;
    bool busy() const override;
    bool updateRunning() const override;

    QList<Package> packages() const override;
    QList<Repository> repositories() const override;

    bool startUpdate(const QStringList &packageIds = QStringList()) override;
    bool rollback(const QStringList &packageIds) override;
    bool removePackages(const QStringList &packageIds) override;
    bool enableRepository(const QString &repositoryId, bool enabled) override;

private:
    State m_state = StateUnknown;
    bool m_updateAvailable = false;
    bool m_updateRunning = false;
    bool m_busy = false;

    QTimer *m_syncTimer = nullptr;
    DBusAgent *m_dbusAgent = nullptr;
    Repository m_repository;
    Package m_package;

    // Well known files and directories for mender state scripts and daemon
    QString m_runtimeDirectory;
    QFileInfo m_currentArtifactInfoFile;
    QFileInfo m_menderDeviceTypeFile;
    QFileInfo m_currentArtifactChangelogFile;

    QFileInfo m_updateStateFile;
    QFileInfo m_updateConfirmFile;
    QFileInfo m_newArtifactInfoFile;
    QFileInfo m_newArtifactChangelogFile;

    QFileInfo m_updateAvailableTimestampFile;
    QFileInfo m_autoUpdateTimestampFile;

    // Private set methods
    void setState(State state);
    void setUpdateAvailable(bool available);
    void setUpdateRunning(bool running);
    void setBusy(bool busy);

    // Helper methods
    QString readArtifactName(const QFileInfo &fileInfo) const;
    QString readDeviceType() const;
    QString readFileContent(const QFileInfo &fileInfo) const;
    QDateTime readTimestampFile(const QFileInfo &fileInfo) const;

    State convertStateString(const QString &stateString) const;
    void readPackageInformation();

    void handleStateLeave(State state);

private slots:
    void onStateEntered(const QString &stateString);
    void onStateLeave(const QString &stateString);

};

#endif // UPDATECONTROLLERMENDER_H
