// SPDX-License-Identifier: GPL-3.0-or-later

/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
*
* Copyright (C) 2013 - 2024, nymea GmbH
* Copyright (C) 2024 - 2025, chargebyte austria GmbH
*
* This file is part of nymea-update-plugin-mender.
*
* nymea-update-plugin-mender is free software: you can redistribute it and/or modify
* it under the terms of the GNU General Public License as published by
* the Free Software Foundation, either version 3 of the License, or
* (at your option) any later version.
*
* nymea-update-plugin-mender is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
* GNU General Public License for more details.
*
* You should have received a copy of the GNU General Public License
* along with nymea-update-plugin-mender. If not, see <https://www.gnu.org/licenses/>.
*
* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

#include "updatecontrollermender.h"

#include <QSysInfo>
#include <QDateTime>

#include <loggingcategories.h>
#include <nymeasettings.h>

UpdateControllerMender::UpdateControllerMender(QObject *parent):
    PlatformUpdateController(parent)
{
    // Well known files and directories for mender state scripts and daemon
    m_runtimeDirectory = "/var/lib/mender/nymea-update";
    m_currentArtifactInfoFile = QFileInfo("/etc/mender/artifact_info");
    m_menderDeviceTypeFile = QFileInfo("/var/lib/mender/device_type");
    m_currentArtifactChangelogFile = QFileInfo("/etc/mender-update-changelog");

    m_updateStateFile = QFileInfo(m_runtimeDirectory + QDir::separator() + "state");
    m_updateConfirmFile = QFileInfo(m_runtimeDirectory + QDir::separator() + "confirm-update");
    m_newArtifactInfoFile = QFileInfo(m_runtimeDirectory + QDir::separator() + "candidate-version");
    m_newArtifactChangelogFile = QFileInfo(m_runtimeDirectory + QDir::separator() + "candidate-changelog");

    m_updateAvailableTimestampFile = QFileInfo(m_runtimeDirectory + QDir::separator() + "update-available-timestamp");
    m_autoUpdateTimestampFile = QFileInfo(m_runtimeDirectory + QDir::separator() + "auto-update-timestamp");

    // Create default repository representing the nymea mender server
    m_repository = Repository("nymea", QT_TR_NOOP("nymea update server"), true);
    enableRepository("nymea", true);

    // Read the mender device type and create package for it.

    QString configurationFileName = "/etc/nymea/mender-update.conf";
    qCDebug(dcPlatformUpdate()) << "Loading configuration from" << configurationFileName;
    QSettings settings(configurationFileName, QSettings::IniFormat, this);
    settings.beginGroup("PackageInformation");
    QString systemName = settings.value("systemName").toString();
    settings.endGroup();

    if (systemName.isEmpty())
        systemName = readDeviceType();

    qCDebug(dcPlatformUpdate()) << "Using system name" << systemName;

    // Create package  for current system
    m_package = Package("system", QT_TR_NOOP("System"));
    m_package.setCanRemove(false);
    m_package.setSummary(QString(QT_TR_NOOP("The %1 operating system.")).arg(systemName));
    m_package.setRollbackAvailable(false);

    // Read the current system package information
    readPackageInformation();

    // Start the dbus adapter for the state script interactions
    m_dbusAgent = new DBusAgent(this);
    connect(m_dbusAgent, &DBusAgent::stateEntered, this, &UpdateControllerMender::onStateEntered);
    connect(m_dbusAgent, &DBusAgent::stateLeave, this, &UpdateControllerMender::onStateLeave);

    if (!m_dbusAgent->connectToDBus(QDBusConnection::SystemBus)) {
        qCWarning(dcPlatformUpdate()) << "Could not register DBus adapter to the system bus. The nymea mender update integration will not be functional.";
        // FIXME: maybe create the update file in order to allow auto updating
        return;
    }

    // Sync timer just in case the dbus message will be missed due to reboots
    m_syncTimer = new QTimer(this);
    m_syncTimer->setInterval(30000);
    m_syncTimer->setSingleShot(false);
    connect(m_syncTimer, &QTimer::timeout, this, [=](){
        State currentState = convertStateString(readFileContent(m_updateStateFile).trimmed());
        if (currentState != StateUnknown) {
            setState(currentState);
        }
    });
    m_syncTimer->start();

    // Set the current state
    setState(convertStateString(readFileContent(m_updateStateFile).trimmed()));
    qCDebug(dcPlatformUpdate()) << "Initialized mender update controller. Current state" << m_state;
}

PlatformUpdateController::UpdateType UpdateControllerMender::updateType() const
{
    return PlatformUpdateController::UpdateTypeSystem;
}

bool UpdateControllerMender::updateManagementAvailable() const
{
    // FIXME: ask systemd if mender is running
    return true;
}

bool UpdateControllerMender::checkForUpdates()
{
    qCDebug(dcPlatformUpdate()) << "Checking for updates...";

    // Note: we should only check for updates if the updater is in Idle state.
    // This makes sure, the update states will not be changed for an update check if there is currently an update
    // running or ready to install. The state should only be changed from the state scripts, except we are in idle and free
    // to do whatever we want.

    if (!updateManagementAvailable()) {
        qCWarning(dcPlatformUpdate()) << "Cannot check for updates. The update manager is not available";
        return false;
    }

    // Only check for updates if there is not an update running currently
    if (m_updateAvailable) {
        qCWarning(dcPlatformUpdate()) << "Cannot check for updates. There is already an update available which has to be installed before checking for new updates. Current state" << m_state;
        return false;
    }

    // Only check for updates if there is not an update running at the moment
    if (updateRunning()) {
        qCWarning(dcPlatformUpdate()) << "Cannot check for updates. There is currently an update running. Current state" << m_state;
        return false;
    }

    // Only check for updates if the updater is not busy. If busy, there is already a check or an update in progress
    if (busy()) {
        qCWarning(dcPlatformUpdate()) << "Cannot check for updates. There update manager is currently busy. Current state" << m_state;
        return false;
    }

    // Only check for updates if the updater is Idle state. Everything else could destroy the update flow due to Sync
    if (m_state != StateIdle) {
        qCWarning(dcPlatformUpdate()) << "Cannot check for updates. This should only be done if the update is in Idle state. Current state" << m_state;
        return false;
    }

    // Call "mender check-update"
    QProcess *process = new QProcess(this);
    connect(process, static_cast<void (QProcess::*)(int, QProcess::ExitStatus)>(&QProcess::finished), this, [process](int exitCode, QProcess::ExitStatus exitStatus){
        process->deleteLater();

        // Verify exit status
        if (exitStatus == QProcess::CrashExit) {
            qCWarning(dcPlatformUpdate()) << "The mender check update command crashed.";
            return;
        }

        // Verify return code
        if (exitCode != 0) {
            qCWarning(dcPlatformUpdate()) << "The mender check update command finished with exit code" << exitCode << process->errorString();
            return;
        }

        qCDebug(dcPlatformUpdate()) << "Check for updates command finished successfully.";
    });

    process->start("mender", { "check-update" });
    return true;
}

bool UpdateControllerMender::busy() const
{
    return m_busy;
}

bool UpdateControllerMender::updateRunning() const
{
    return m_updateRunning;
}

QList<Package> UpdateControllerMender::packages() const
{
    return { m_package };
}

QList<Repository> UpdateControllerMender::repositories() const
{
    return { m_repository };
}

bool UpdateControllerMender::startUpdate(const QStringList &packageIds)
{
    qCDebug(dcPlatformUpdate()) << "Starting to update" << packageIds;

    // Create the update confirm file in order to inform the state script that the update can be installed
    QFile updateConfirmFile(m_updateConfirmFile.absoluteFilePath());
    if (!updateConfirmFile.open(QIODevice::WriteOnly)) {
        qCWarning(dcPlatformUpdate()) << "Could not create confirmation file" << updateConfirmFile.fileName() << updateConfirmFile.errorString();
        setUpdateRunning(false);
        return false;
    }

    // Write something just to make sure the file will be written
    updateConfirmFile.write(" ");
    updateConfirmFile.close();
    setUpdateRunning(true);
    return true;
}

bool UpdateControllerMender::rollback(const QStringList &packageIds)
{
    Q_UNUSED(packageIds)
    qCWarning(dcPlatformUpdate()) << "Manual rollback is not supported.";
    return false;
}

bool UpdateControllerMender::removePackages(const QStringList &packageIds)
{
    Q_UNUSED(packageIds)
    qCWarning(dcPlatformUpdate()) << "Removing is not supported for this update plugin.";
    return false;
}

bool UpdateControllerMender::enableRepository(const QString &repositoryId, bool enabled)
{
    if (!enabled) {
        qCWarning(dcPlatformUpdate()) << "Disable the repository is not supported for this plugin.";
        return false;
    }

    qCDebug(dcPlatformUpdate()) << "Enable repository" << repositoryId;
    return true;
}

void UpdateControllerMender::setState(UpdateControllerMender::State state)
{
    if (m_state == state)
        return;

    qCDebug(dcPlatformUpdate()) << "Entering " << state;
    m_state = state;

    switch (state) {
    case StateUnknown:
        qCWarning(dcPlatformUpdate()) << "Unknown mender state. Please report a bug!";
        break;
    case StateIdle:
        setUpdateAvailable(false);
        setUpdateRunning(false);
        setBusy(false);
        readPackageInformation();
        break;
    case StateSync:
        setUpdateRunning(false);
        setUpdateAvailable(false);
        setBusy(true);
        break;
    case StateUpdateAvailable:
        setUpdateRunning(false);
        setBusy(false);
        readPackageInformation();
        break;
    case StateDownload:
    case StateArtifactInstall:
    case StateArtifactReboot:
    case StateArtifactCommit:
    case StateArtifactRollback:
    case StateArtifactRollbackReboot:
        setUpdateRunning(true);
        setUpdateAvailable(false);
        setBusy(true);
        break;
    case StateArtifactFailure:
        qCWarning(dcPlatformUpdate()) << "Mender entered the artifact failure state.";
        setUpdateRunning(false);
        setUpdateAvailable(false);
        break;
    }
}

void UpdateControllerMender::setUpdateAvailable(bool available)
{
    if (m_updateAvailable == available)
        return;

    qCDebug(dcPlatformUpdate()) << "Update available changed" << available;
    m_updateAvailable = available;
}

void UpdateControllerMender::setUpdateRunning(bool running)
{
    if (m_updateRunning == running)
        return;

    qCDebug(dcPlatformUpdate()) << "Update running changed" << running;
    m_updateRunning = running;
    emit updateRunningChanged();
}

void UpdateControllerMender::setBusy(bool busy)
{
    if (m_busy == busy)
        return;

    qCDebug(dcPlatformUpdate()) << "Update busy changed" << busy;
    m_busy = busy;
    emit busyChanged();
}

QString UpdateControllerMender::readArtifactName(const QFileInfo &fileInfo) const
{
    qCDebug(dcPlatformUpdate()) << "Reading current artifact name from" << fileInfo.absoluteFilePath();
    QFile artifactInfoFile(fileInfo.absoluteFilePath());
    if (!artifactInfoFile.open(QIODevice::ReadOnly | QIODevice::Text)) {
        qCWarning(dcPlatformUpdate()) << "Could not open artifact file" << artifactInfoFile.fileName() << artifactInfoFile.errorString();
        return QString();
    }

    // Parse file line by line
    QString artifactName;
    QTextStream stream(&artifactInfoFile);
    while (!stream.atEnd()) {
        QString line = stream.readLine();
        if (line.startsWith("artifact_name")) {
            QStringList tokens = line.split("=");
            //qCDebug(dcPlatformUpdate()) << tokens;
            if (tokens.count() == 2) artifactName = tokens.at(1);
        }
    }

    artifactInfoFile.close();

    qCDebug(dcPlatformUpdate()) << "Artifact name:" << artifactName;
    return artifactName;
}

QString UpdateControllerMender::readDeviceType() const
{
    QString menderDeviceType;
    QString deviceTypeString = readFileContent(m_menderDeviceTypeFile).trimmed();
    if (deviceTypeString.isEmpty()) {
        qCWarning(dcPlatformUpdate()) << "Failed to fetch device type from" << m_menderDeviceTypeFile.absoluteFilePath();
        return menderDeviceType;
    }

    QStringList tokes = deviceTypeString.split("=");
    if (tokes.count() != 2) {
        qCWarning(dcPlatformUpdate()) << "Failed to parse device type string" << deviceTypeString << "from" << m_menderDeviceTypeFile.absoluteFilePath();
        return menderDeviceType;
    }

    menderDeviceType = tokes.at(1);
    qCDebug(dcPlatformUpdate()) << "Mender device type" << menderDeviceType;

    menderDeviceType = menderDeviceType.replace('-', ' ');
    qCDebug(dcPlatformUpdate()) << "Using prettified mender device type:" << menderDeviceType;
    return menderDeviceType;
}

QString UpdateControllerMender::readFileContent(const QFileInfo &fileInfo) const
{
    if (!fileInfo.exists()) {
        qCDebug(dcPlatformUpdate()) << "Cannot read file content of" << fileInfo.absoluteFilePath() << "because the file does not exist.";
        return QString();
    }

    QFile file(fileInfo.absoluteFilePath());
    if (!file.open(QIODevice::ReadOnly | QIODevice::Text)) {
        qCWarning(dcPlatformUpdate()) << "Could not open file" << file.fileName() << ":" << file.errorString();
        return QString();
    }

    // Parse file line by line
    QTextStream stream(&file);
    QString content = stream.readAll();
    file.close();
    return content;
}

QDateTime UpdateControllerMender::readTimestampFile(const QFileInfo &fileInfo) const
{
    if (!fileInfo.exists()) {
        qCDebug(dcPlatformUpdate()) << "Cannot read file content of" << fileInfo.absoluteFilePath() << "because the file does not exist.";
        return QDateTime();
    }

    QString content = readFileContent(fileInfo);
    if (content.isEmpty())
        return QDateTime();

    // Convert string to timestamp
    bool valueOk = false;
    quint64 unixTimeStamp = content.toLongLong(&valueOk);
    if (!valueOk) {
        qCWarning(dcPlatformUpdate()) << "Failed to convert timestamp file content to uint64" << content << fileInfo.fileName();
        return QDateTime();
    }

    return QDateTime::fromMSecsSinceEpoch(unixTimeStamp * 1000);
}

UpdateControllerMender::State UpdateControllerMender::convertStateString(const QString &stateString) const
{
    State state = StateUnknown;
    if (stateString == "Idle") {
        state = StateIdle;
    } else if (stateString == "Sync") {
        state = StateSync;
    } else if (stateString == "Download") {
        state = StateDownload;
    } else if (stateString == "UpdateAvailable") {
        state = StateUpdateAvailable;
    } else if (stateString == "ArtifactInstall") {
        state = StateArtifactInstall;
    } else if (stateString == "ArtifactReboot") {
        state = StateArtifactReboot;
    } else if (stateString == "ArtifactCommit") {
        state = StateArtifactCommit;
    } else if (stateString == "ArtifactRollback") {
        state = StateArtifactRollback;
    } else if (stateString == "ArtifactRollbackReboot") {
        state = StateArtifactRollbackReboot;
    } else if (stateString == "ArtifactFailure") {
        state = StateArtifactFailure;
    } else {
        qCWarning(dcPlatformUpdate()) << "Unrecognized state string" << stateString << ". The state will be unknown.";
    }

    return state;
}

void UpdateControllerMender::readPackageInformation()
{
    // Read the update information
    bool packageHasChanged = false;

    // Set current installed version
    QString installedVersion = readArtifactName(m_currentArtifactInfoFile);
    QString installedChangelog = readFileContent(m_currentArtifactChangelogFile);
    if (m_package.installedVersion() != installedVersion)
        packageHasChanged = true;

    m_package.setInstalledVersion(installedVersion);
    m_package.setChangelog(installedChangelog);

    // Load candidate information if available
    QString candidateVersion;
    QString candidateChangelog;
    if (m_newArtifactInfoFile.exists())
        candidateVersion = readArtifactName(m_newArtifactInfoFile).trimmed();

    if (m_newArtifactChangelogFile.exists())
        candidateChangelog = readFileContent(m_newArtifactChangelogFile);

    // If the current candidate version is different than the new artifact verion (even if empty)
    if (m_package.candidateVersion() != candidateVersion) {
        m_package.setCandidateVersion(candidateVersion);
        if (!candidateChangelog.isEmpty()) {
            m_package.setChangelog(candidateChangelog);
        }
        packageHasChanged = true;
    }

    // Check if an update is available
    if (m_package.candidateVersion() == m_package.installedVersion() || m_package.candidateVersion().isEmpty()) {
        if (m_package.updateAvailable()) {
            m_package.setUpdateAvailable(false);
            packageHasChanged = true;
        }
    }

    if (!m_package.candidateVersion().isEmpty() && m_package.installedVersion() != m_package.candidateVersion()) {
        if (!m_package.updateAvailable()) {
            m_package.setUpdateAvailable(true);
            packageHasChanged = true;
        }
    }

    // Read the timestamp files if they exist
    QDateTime updateAvailableSinceDateTime;
    QDateTime autoUpdateDateTime;
    if (m_updateAvailableTimestampFile.exists())
        updateAvailableSinceDateTime = readTimestampFile(m_updateAvailableTimestampFile);

    if (m_autoUpdateTimestampFile.exists())
        autoUpdateDateTime = readTimestampFile(m_autoUpdateTimestampFile);

    setUpdateAvailable(m_package.updateAvailable());

    // Tell nymea the package has changed
    if (packageHasChanged) {
        qCDebug(dcPlatformUpdate()) << "Package information changed" << m_package.packageId() << m_package.displayName();
        qCDebug(dcPlatformUpdate()) << "- Installed version:" << m_package.installedVersion();
        qCDebug(dcPlatformUpdate()) << "- Update available:" << m_package.updateAvailable();

        if (m_package.updateAvailable())
            qCDebug(dcPlatformUpdate()) << "- Candidate version:" << m_package.candidateVersion();

        if (updateAvailableSinceDateTime.isValid() && autoUpdateDateTime.isValid()) {
            qCDebug(dcPlatformUpdate()) << "- Update available since" << updateAvailableSinceDateTime.toString("dd.MM.yyyy hh:mm:ss");
            qCDebug(dcPlatformUpdate()) << "- Auto update will be triggert at" << autoUpdateDateTime.toString("dd.MM.yyyy hh:mm:ss");
        }

        emit packageChanged(m_package);
    }
}

void UpdateControllerMender::handleStateLeave(UpdateControllerMender::State state)
{
    qCDebug(dcPlatformUpdate()) << "Leaving" << state;
}

void UpdateControllerMender::onStateEntered(const QString &stateString)
{
    setState(convertStateString(stateString));
}

void UpdateControllerMender::onStateLeave(const QString &stateString)
{
    handleStateLeave(convertStateString(stateString));
}
