// SPDX-License-Identifier: LGPL-3.0-or-later

/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
*
* Copyright (C) 2013 - 2024, nymea GmbH
* Copyright (C) 2024 - 2025, chargebyte austria GmbH
*
* This file is part of nymea.
*
* nymea is free software: you can redistribute it and/or
* modify it under the terms of the GNU Lesser General Public License
* as published by the Free Software Foundation, either version 3
* of the License, or (at your option) any later version.
*
* nymea is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
* GNU Lesser General Public License for more details.
*
* You should have received a copy of the GNU Lesser General Public License
* along with nymea. If not, see <https://www.gnu.org/licenses/>.
*
* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

#ifndef JSONRPCSERVERIMPLEMENTATION_H
#define JSONRPCSERVERIMPLEMENTATION_H

#include "jsonrpc/jsonrpcserver.h"
#include "jsonrpc/jsonhandler.h"
#include "transportinterface.h"
#include "usermanager/usermanager.h"

#include "types/thingclass.h"
#include "types/action.h"
#include "types/event.h"

#include <QObject>
#include <QVariantMap>
#include <QString>
#include <QSslConfiguration>

class Thing;

namespace nymeaserver {

class JsonRPCServerImplementation: public JsonHandler, public JsonRPCServer
{
    Q_OBJECT
public:
    JsonRPCServerImplementation(const QSslConfiguration &sslConfiguration = QSslConfiguration(), QObject *parent = nullptr);

    // JsonHandler API implementation
    QString name() const override;
    Q_INVOKABLE JsonReply *Hello(const QVariantMap &params, const JsonContext &context);
    Q_INVOKABLE JsonReply *Introspect(const QVariantMap &params) const;
    Q_INVOKABLE JsonReply *Version(const QVariantMap &params) const;
    Q_INVOKABLE JsonReply *SetNotificationStatus(const QVariantMap &params, const JsonContext &context);

    Q_INVOKABLE JsonReply *CreateUser(const QVariantMap &params);
    Q_INVOKABLE JsonReply *Authenticate(const QVariantMap &params, const JsonContext &context);
    Q_INVOKABLE JsonReply *RequestPushButtonAuth(const QVariantMap &params, const JsonContext &context);
    Q_INVOKABLE JsonReply *KeepAlive(const QVariantMap &params);

signals:
    void PushButtonAuthFinished(const QUuid &clientId, const QVariantMap &params);

    // Server API
public:
    void registerTransportInterface(TransportInterface *interface);
    void unregisterTransportInterface(TransportInterface *interface);

    bool registerHandler(JsonHandler *handler) override;
    bool registerExperienceHandler(JsonHandler *handler, int majorVersion, int minorVersion) override;

private:
    QHash<QString, JsonHandler *> handlers() const;

    void sendResponse(TransportInterface *interface, const QUuid &clientId, int commandId, const QVariantMap &params = QVariantMap(), const QString &deprecationWarning = QString());
    void sendErrorResponse(TransportInterface *interface, const QUuid &clientId, int commandId, const QString &error);
    void sendUnauthorizedResponse(TransportInterface *interface, const QUuid &clientId, int commandId, const QString &error);

    void processJsonPacket(TransportInterface *interface, const QUuid &clientId, const QByteArray &data);

private slots:
    void setup();

    void clientConnected(const QUuid &clientId);
    void clientDisconnected(const QUuid &clientId);

    void processData(const QUuid &clientId, const QByteArray &data);

    void sendNotification(const QVariantMap &params);
    void sendClientNotification(const QUuid &clientId, const QVariantMap &params);

    void asyncReplyFinished();

    void onPushButtonAuthFinished(int transactionId, bool success, const QByteArray &token);

private:
    QVariantMap m_api;
    QHash<JsonHandler *, QString> m_experiences;
    QHash<QString, JsonHandler *> m_handlers;
    QHash<JsonReply *, TransportInterface *> m_asyncReplies;

    QHash<QUuid, TransportInterface *> m_clientTransports;
    QHash<QUuid, QByteArray> m_clientBuffers;
    QHash<QUuid, QStringList> m_clientNotifications;
    QHash<QUuid, QLocale> m_clientLocales;
    QHash<QUuid, QByteArray> m_clientTokens;
    QHash<int, QUuid> m_pushButtonTransactions;
    QHash<QUuid, QTimer *> m_newConnectionWaitTimers;

    QHash<QString, JsonReply *> m_pairingRequests;

    int m_notificationId;

    QTimer m_connectionLockdownTimer;

    QString formatAssertion(const QString &targetNamespace, const QString &method, QMetaMethod::MethodType methodType, JsonHandler *handler, const QVariantMap &data) const;
};

}

#endif // JSONRPCSERVERIMPLEMENTATION_H

